/*
 * Copyright (C) 2000 - 2024 Silverpeas
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * As a special exception to the terms and conditions of version 3.0 of
 * the GPL, you may redistribute this Program in connection with Free/Libre
 * Open Source Software ("FLOSS") applications as described in Silverpeas's
 * FLOSS exception.  You should have received a copy of the text describing
 * the FLOSS exception, and it is also available here:
 * "https://www.silverpeas.org/legal/floss_exception.html"
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
/* $Id$ */
package org.silverpeas.core.contribution.publication.model;

import org.silverpeas.core.ResourceReference;

import java.io.Serializable;
import java.util.Objects;

/**
 * A reference to a publication in Silverpeas.
 *
 * @author Nicolas Eysseric
 * @version 1.0
 */
public class PublicationPK extends ResourceReference implements Serializable {
  private static final long serialVersionUID = -6957633959517435029L;

  public PublicationPK(String id) {
    super(id);
  }

  public PublicationPK(String id, String space, String componentName) {
    super(id, componentName);
    setSpace(space);
  }

  public PublicationPK(String id, String componentId) {
    super(id, componentId);
  }

  public PublicationPK(String id, ResourceReference pk) {
    this(id, pk.getSpace(), pk.getInstanceId());
  }

  /**
   * Return the object root table name
   *
   * @return the root table name of the object
   * @since 1.0
   */
  @Override
  public String getRootTableName() {
    return "Publication";
  }

  /**
   * Return the object table name
   *
   * @return the table name of the object
   * @since 1.0
   */
  @Override
  public String getTableName() {
    return "SB_Publication_Publi";
  }

  /**
   * Check if this reference refers the same publication than the specified one. A publication is
   * uniquely identified by its identifier, whatever the component instance it belongs to. Two
   * instances in different component instances but with the same identifier refer the same
   * publication; this means the same publication can be referred from different location.
   *
   * @param obj the object to compare to this PublicationPK
   * @return true if other is equals to this object
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (!(obj instanceof PublicationPK)) {
      return false;
    }
    PublicationPK other = (PublicationPK) obj;
    return Objects.equals(id, other.id);
  }

  /**
   * Returns a hash code for the key. In opposite to the default behavior of the
   * {@link ResourceReference}, it is computed only on the publication identifier.
   *
   * @return A hash code for this object
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    return Objects.hash(id);
  }
}
